<?php
declare(strict_types=1);
namespace MMNewmedia\BiPRO\Norm410\Decorator;

use MMNewmedia\BiPRO\Entity\Nachrichten\BiPROException;
use MMNewmedia\BiPRO\Entity\Nachrichten\Status;
use MMNewmedia\BiPRO\Entity\Nachrichten\Meldung;

/**
 * AuthDecorator Klasse (utf-8)
 * Überprüft die Credentials und führt dann die vom Webservice angefragten Methoden ab
 * 
 * @author MarcelMaaß <info@mm-newmedia.de>
 * @copyright 2016 MM Newmedia <http://www.mm-newmedia.de>
 * @package de.mmnewmedia.bipro
 * @subpackage norm410.decorator
 * @since 2016-03-03
 * @version
 */
class AuthDecorator 
{	
	/**
	 * Name der Klasse, welche die Webservice Methoden enthält
	 * @var string
	 */
	protected $sClassName = '';
	
	/**
	 * Ist der User authentifiziert
	 * @var boolean
	 */
	protected $bAuthenticated = false;
	
	/**
	 * Prüft Benutzernamen und Passwort und führt dann die übergebene Methode aus
	 * 
	 * @param 	string 	$sMethodName Name der auszuführenden Methode
	 * @param 	array 	$aArguments	 Argumente für die auszufhrende Methode
	 * @return	object
	 * @throws \SoapFault
	 */
	public function __call(string $sMethodName, array $aArguments = []) 
	{
		if (!method_exists($this->sClassName, $sMethodName)) {
			throw new \SoapFault(
				'Server',
				sprintf(
					'Die Methode %s existiert nicht.',
					$sMethodName
				)
			);
		}
		
		if (!$this->getAuthenticated()) {
		    
		    // Eigenschaften für das Meldungsobjekt
		    $oArtID = new \SoapVar('Fehler', XSD_STRING, '', '', 'ArtID', 'http://www.bipro.net/namespace/nachrichten');
		    $oMeldungID = new \SoapVar('00961', XSD_STRING, '', '', 'MeldungID', 'http://www.bipro.net/namespace/nachrichten');
		    $oText = new \SoapVar('Credentials (Username, Password, OTP) ungültig', XSD_STRING, '', '', 'Text', 'http://www.bipro.net/namespace/nachrichten');
		    
		    $oMeldung = (new Meldung())
		        ->setArtID($oArtID)
		        ->setMeldungID($oMeldungID)
		        ->setText($oText)
		        ->encode();
		    
	        // Eigenschaften für das Status Objekt
	        $oProzessID = new \SoapVar(getmypid(), XSD_STRING, '', '', 'ProzessID', 'http://www.bipro.net/namespace/nachrichten');
	        $oZeitstempel = new \SoapVar((new \DateTime())->format('Y-m-d H:i:s'), XSD_STRING, '', '', 'Zeitstempel', 'http://www.bipro.net/namespace/nachrichten');
	        $oStatusID = new \SoapVar('NOK', XSD_STRING, '', '', 'StatusID', 'http://www.bipro.net/namespace/nachrichten');
		    
		    $oStatus = (new Status())
		        ->setProzessID($oProzessID)
		        ->setZeitstempel($oZeitstempel)
		        ->setStatusID($oStatusID)
		        ->setMeldung($oMeldung)
		        ->encode();
		    
		    // Eigenschaften für die BiPRO Exception
		    $oBiPROVersion = new \SoapVar('2.1.0.1.0', XSD_STRING, '', '', 'BiPROVersion', 'http://www.bipro.net/namespace/nachrichten');
	        
			$oException = (new BiPROException())
			    ->setBiPROVersion($oBiPROVersion)
			    ->setStatus($oStatus)
			    ->encode();
			
			throw new \SoapFault(
			    'Server', 
			    'Authentication Error', 
			    null, 
			    $oException
			);
		}
		
		return call_user_func_array(
			[$this->sClassName, $sMethodName], 
			$aArguments
		);
	}
	
	/**
	 * Wird automatisch vom empfangenen Soap Header aufgerufen und prüft die Credentials
	 * 
	 * @param object $WsseHeader
	 */
	public function Security($oSecurity) : void
	{
		if ($oSecurity->UsernameToken->Username == 'Foo' && $oSecurity->UsernameToken->Password == 'Bar') {
			$this->setAuthenticated(true);
		}
	}
	
	/**
	 * Liefert den Klassennamen der Klasse, die die Webservice Methoden beinhaltet
	 * 
	 * @return string
	 */
	public function getClassName() : string
	{
		return $this->sClassName;
	}
	
	/**
	 * Setzt den Klassennamen der Klasse, die die Webservice Methoden beinhaltet
	 * 
	 * @param string $sClassName
	 * @return AuthDecorator
	 */
	public function setClassName(string $sClassName) : AuthDecorator
	{
		$this->sClassName = $sClassName;
		return $this;
	}
	
	/**
	 * Liefert das Flag, ob ein Request mit den gelieferten Credentials positiv authentifiziert wurde
	 * 
	 * @return boolean
	 */
	public function getAuthenticated() : bool
	{
		return $this->bAuthenticated;
	}
	
	/**
	 * Setzt das Flag, ob ein Request mit den gelieferten Credentials positiv authentifiziert wurde
	 * 
	 * @param boolean $bAuthenticated
	 * @return AuthDecorator
	 */
	public function setAuthenticated(bool $bAuthenticated) : AuthDecorator
	{
		$this->bAuthenticated = $bAuthenticated;
		return $this;
	}
}