<?php
declare(strict_types=1);
namespace MMNewmedia\BiPRO\Entity\Nachrichten;

use MMNewmedia\BiPRO\Entity\Basis\Objekt;
use MMNewmedia\BiPRO\Entity\BiPROEntity;
use MMNewmedia\BiPRO\Entity\NullProperties;

/**
 * Meldung Entity nach BiPRO Norm (utf-8)
 * Ermoeglicht die Erweiterung einer Statusmeldung um zusaetzliche Informationen (z.B. Fehlermeldungen oder Hinweise), insbesondere
 * dann, wenn eine synchrone Verarbeitung nicht mit einem Ergebnis abgeschlossen werden konnte. In Anhängigkeit zu Status.statusid 
 * MUSS oder KANN das Objekt übermittelt werden.
 * 
 * @author Marcel Maaß <info@mm-newmedia.de>
 * @copyright 2016 MM Newmedia <http://www.mm-newmedia.de>
 * @package de.mmnewmedia.bipro
 * @subpackage entity.nachrichten
 * @since 2018-10-07
 * @version 2.6.0
 */
class Meldung extends Objekt implements BiPROEntity
{
    use NullProperties;
    
    /**
     * Art der Meldung
     * @var \SoapVar
     * @internal ST_Meldungsart
     */
    protected $ArtID;
    
    /**
     * Eindeutige Identifizierung der Meldung
     * @var \SoapVar
     * @internal ST_Meldung
     */
    protected $MeldungID;
    
    /**
     * Eindeutige Referenz auf das fehlerhafte Objekt und die Objekt-Eigenschaft mittels XPath
     * @var \SoapVar
     * @internal ST_Text
     */
    protected $Text;
    
    /**
     * Alphanumerische eindeutige Kennzeichnung der Meldung bzw. des Fehlers oder Hinweises. Anhand der Nummer wird insbesondere die
     * nichttechnische Kommunikation zwischen Nutzer und Anbieter erleichtert.
     * @var \SoapVar
     * @internal ST_Text30
     */
    protected $Quelle;
    
    /**
     * Liefert die ArtID
     * @return \SoapVar
     */
    protected $Identifikationsnummer;
    
    /**
     * Liefert die Art ID
     * @return \SoapVar|NULL
     */
    public function getArtID() : ?\SoapVar {
        return $this->ArtID;
    }
    
    /**
     * Setzt die ArtID
     * @param \SoapVar $oArtID
     * @return Meldung
     */
    public function setArtID(\SoapVar $oArtID) : Meldung
    {
        $this->ArtID = $oArtID;
        return $this;
    }
    
    /**
     * Liefert die MeldungID
     * @return \SoapVar|NULL
     */
    public function getMeldungID() : ?\SoapVar
    {
        return $this->MeldungID;
    }
    
    /**
     * Setzt die MeldungID
     * @param \SoapVar $oMeldungID
     * @return Meldung
     */
    public function setMeldungID(\SoapVar $oMeldungID) : Meldung
    {
        $this->MeldungID = $oMeldungID;
        return $this;
    }
    
    /**
     * Liefert den Text der Meldung
     * @return \SoapVar|NULL
     */
    public function getText() : ?\SoapVar
    {
        return $this->Text;
    }
    
    /**
     * Setzt den Text der Meldung
     * @param \SoapVar $oText
     * @return Meldung
     */
    public function setText(\SoapVar $oText) : Meldung
    {
        $this->Text = $oText;
        return $this;
    }
    
    /**
     * Liefert die Quelle
     * @return string|NULL
     */
    public function getQuelle() : ?string
    {
        return $this->Quelle;
    }
    
    /**
     * Setzt die Quelle
     * @param \SoapVar $oQuelle
     * @return Meldung
     */
    public function setQuelle(\SoapVar $oQuelle) : Meldung
    {
        $this->Quelle = $oQuelle;
        return $this;
    }
    
    /**
     * Liefert die Identifikationsnummer
     * @return \SoapVar|NULL
     */
    public function getIdentifikationsnummer() : ?\SoapVar {
        return $this->Identifikationsnummer;
    }
    
    /**
     * @see \MMNewmedia\BiPRO\Entity\BiPROEntity::encode()
     */
    public function encode() : \SoapVar
    {
        // Entfernen von NULL Eigenschaften über das NullProperties Trait
        $this->remove();
        
        $oEncodedObject = new \SoapVar(
            $this,
            SOAP_ENC_OBJECT,
            '',
            '',
            'Meldung',
            'http://www.bipro.net/namespace/nachrichten'
        );
        
        return $oEncodedObject;
    }
}