<?php
declare(strict_types=1);
namespace MMNewmedia\BiPRO\Entity\Nachrichten;

use MMNewmedia\BiPRO\Entity\Basis\Objekt;
use MMNewmedia\BiPRO\Entity\BiPROEntity;
use MMNewmedia\BiPRO\Entity\NullProperties;

/**
 * Dient der Abfrage oder Übermittlung des Status eines laufenden oder abgeschlossenen Prozesses.
 * 
 * @author Marcel Maaß <info@mm-newmedia.de>
 * @copyright 2016 MM Newmedia <http://www.mm-newmedia.de>
 * @package de.mmnewmedia.bipro
 * @subpackage entity.nachrichten
 * @since 2018-10-07
 * @version 2.6.0
 */
class Status extends Objekt implements BiPROEntity
{
    use NullProperties;
    
    /**
     * Eindeutige Identifizierung des fachlichen Prozesses beim Provider.
     * @var null|\SoapVar
     * @internal ST_ProzessID
     */
    protected $ProzessID;
    
    /**
     * Aktuelles Datum und Uhrzeit zum Bearbeitungszeitpunkt.
     * @var null|\SoapVar
     * @internal ST_Zeitstempel
     */
    protected $Zeitstempel;
    
    /**
     * Eindeutige Identifizierung der Statusmeldung.
     * @var null|\SoapVar
     * @internal ST_Status
     */
    protected $StatusID;
    
    /**
     * Zeitpunkt bis zu dem die zum Vorgang gehörigen Daten beim Anbieter vorgehalten werden.
     * @var null|\SoapVar
     * @internal ST_Zeitstempel
     */
    protected $Gueltigkeitsende;
    
    /**
     * Ein Teilprozess wird regelmäßig technisch synchron durchlaufen. Nach einem Request des Service Consumers folgt ein Response des 
     * Service Providers. Damit ist dieser Prozess abgeschlossen. Ein technisch synchron übermittelter Geschäftsvorgang kann jedoch nicht
     * immer fallabschließend bearbeitet werden. Das Ende der Bearbeitung stimmt entsprechend nicht immer mit dem Ende des Teilprozesses überein.
     * In diesem Fall handelt es sich um einen schwebenden Vorgang. Zur Kennzeichnung, dass der Vorgang schwebend ist, MUSS das Attribut Schwebe
     * mit dem Wert true gefüllt werden. Dies ist für den Service Consumer zugleich in Indikator, an dem er erkennen kann, dass möglicherweise
     * ein Tracking des Vorgangs über einen dritten Business Service (z.B. N430) erfolgen kann.
     * @var null|\SoapVar
     * @internal ST_JaNein
     */
    protected $Schwebe;
    
    /**
     * Meldung
     * @var null|\SoapVar
     * @internal CT_Meldung
     */
    protected $Meldung;
    
    /**
     * Liefert die Prozess ID
     * @return \SoapVar|NULL
     */
    public function getProzessID() : ?\SoapVar
    {
        return $this->ProzessID;
    }
    
    /**
     * Setzt die Prozess ID
     * @param \SoapVar $oProzessID
     * @return Status
     */
    public function setProzessID(\SoapVar $oProzessID) : Status
    {
        $this->ProzessID = $oProzessID;
        return $this;
    }
    
    /**
     * Liefert den Zeitstempel
     * @return \SoapVar|NULL
     */
    public function getZeitstempel() : ?\SoapVar
    {
        return $this->Zeitstempel;
    }
    
    /**
     * Setzt den Zeitstempel
     * @param \SoapVar $oZeitstempel
     * @return Status
     */
    public function setZeitstempel(\SoapVar $oZeitstempel) : Status
    {
        $this->Zeitstempel = $oZeitstempel;
        return $this;
    }
    
    /**
     * Liefert die Status ID
     * @return \SoapVar|NULL
     */
    public function getStatusID() : ?\SoapVar
    {
        return $this->StatusID;
    }
    
    /**
     * Setzt die Status ID
     * @param \SoapVar $oStatusID
     * @return Status
     */
    public function setStatusID(\SoapVar $oStatusID) : Status
    {
        $this->StatusID = $oStatusID;
        return $this;
    }
    
    /**
     * Liefert das Gültigkeitsende
     * @return \SoapVar|NULL
     */
    public function getGueltigkeitsende() : ?\SoapVar
    {
        return $this->Gueltigkeitsende;
    }
    
    /**
     * Setzt das Gültigkeitesende
     * @param \SoapVar $oGueltigkeitsende
     * @return Status
     */
    public function setGueltigkeitsende(\SoapVar $oGueltigkeitsende) : Status
    {
        $this->Gueltigkeitsende = $oGueltigkeitsende;
        return $this;
    }
    
    /**
     * Liefert das Kennzeichen für die Schwebe
     * @return \SoapVar|NULL
     */
    public function getSchwebe() : ?\SoapVar
    {
        return $this->Schwebe;
    }
    
    /**
     * Setzt das Kennzeichen für die Schwebe
     * @param \SoapVar $oSchwebe
     * @return Status
     */
    public function setSchwebe(\SoapVar $oSchwebe) : Status
    {
        $this->Schwebe = $oSchwebe;
        return $this;
    }
    
    /**
     * Liefert das Meldungsobjekt
     * @return \SoapVar|NULL
     */
    public function getMeldung() : ?\SoapVar
    {
        return $this->Meldung;
    }
    
    /**
     * Setzt das Meldungsobjekt
     * @param \SoapVar $oMeldung
     * @return Status
     */
    public function setMeldung(\SoapVar $oMeldung) : Status
    {
        $this->Meldung = $oMeldung;
        return $this;
    }
    
    /**
     * @see \MMNewmedia\BiPRO\Entity\BiPROEntity::encode()
     */
    public function encode() : \SoapVar
    {
        // Entfernen von NULL Eigenschaften über das NullProperties Trait
        $this->remove();
        
        $oEncodedObject = new \SoapVar(
            $this,
            SOAP_ENC_OBJECT,
            '',
            '',
            'Status',
            'http://www.bipro.net/namespace/nachrichten'
        );
        
        return $oEncodedObject;
    }
}